// Marching cubes tutorial example code by paradox / vivid (Nicholas Nash)
// email: nnash@stokes2.thphys.may.ie or nash@cfxweb.net with questions/comments.

// Sorry about some literals in this code, enjoy!
#include <windows.h>
#include <math.h>
#include <gl/gl.h>
#include <gl/glu.h>
#include "glapp.h"
#include "mcubes.cpp"

typedef unsigned char byte;

// This is the surface we give to CalcMesh
class SphereSurface
{
   int nSpheres, nCells; 
   float minPotential;
   Point3D *coords, stepSize, spaceSize;
public:
	float *potentialBuf;
	SphereSurface(int nSpheres, int radius, int nCells, Point3D spaceSize, Point3D *coords);
	float getPotential(Point3D p);
	bool outside(Point3D p);
	Point3D getIntersection(Point3D p1, Point3D p2);
   ~SphereSurface();
};

void setGLOptions();

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{ 
   if(!glApp.makeWindow(hInstance, "classname", "Marching Cubes Tutorial")) return 0;
  	
   // Some hard-coded data:
   const int nCells = 19, cubeSide = 9, nBlobs = 6, blobRadius = 18;
   Point3D spaceSize(cubeSide * nCells, cubeSide * nCells, cubeSide * nCells); 
   Point3D *coords = new Point3D[nBlobs], *dirs = new Point3D[nBlobs];
   Point3D *mesh = new Point3D[5 * nCells * nCells * nCells]; 
   SphereSurface surf(nBlobs, blobRadius, nCells, spaceSize, coords);
	CalcMesh <SphereSurface> surfMesh(spaceSize, nCells, &surf, mesh);
   
   setGLOptions();
   for(int i = 0; i < nBlobs; i++) 
   {
      coords[i] = Point3D(0, 0, 0);
      dirs[i] = Point3D((rand() % 6) - 3, (rand() % 4) - 2, (rand() % 6) - 3); 
   }
   MSG msg;		
	while(msg.message != WM_QUIT)
	{
		if(PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE))
		{
			if(GetMessage(&msg, NULL, 0, 0))
			{
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
		}
      for(i = 0; i < nBlobs; i++) 
      { 
         coords[i].x += dirs[i].x; 
         coords[i].y += dirs[i].y; 
         coords[i].z += dirs[i].z; 
         if(coords[i].x <= -45 || coords[i].x > 45) dirs[i].x *= -1;
         if(coords[i].y <= -45 || coords[i].y > 45) dirs[i].y *= -1;
         if(coords[i].z <= -45 || coords[i].z > 40) dirs[i].z *= -1;  
      }
      surfMesh.reset();
      memset(surf.potentialBuf, 0, nCells * nCells * nCells * sizeof(float));
      for(i = 0; i < nBlobs; i++) 
      {     
         // This is the surface generation code.
         surfMesh.generate(
         surfMesh.getCube(coords[i].x + blobRadius - 1, coords[i].y, coords[i].z),
         coords[i].y, coords[i].z);
      }  
	   static float theta = 0, phi = 0; 
      theta += 3.5;
      phi += 2.5;
      glLoadIdentity();	
      glClear(GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT);
      glTranslatef(0.0f, 0.0f, -160); 
      glRotatef(theta, 0.0f, 1.0f, 0.0f);
      glRotatef(phi, 0.2f, 0.0f, 0.1f);
      glBegin(GL_TRIANGLES);
         for(i = 0; i < surfMesh.nVerts; i++) glVertex3i(surfMesh.mesh[i].x, surfMesh.mesh[i].y, surfMesh.mesh[i].z);
      glEnd();
      glApp.flip();
   }
   delete [] coords;
	delete [] dirs;
   delete [] mesh;
   return 0;
}

SphereSurface::SphereSurface(int nSpheres, int radius, int nCells, Point3D spaceSize, Point3D *coords)
{
	this->nSpheres = nSpheres;
	this->nCells = nCells;	
	this->stepSize = stepSize;
	this->coords = coords;
	this->spaceSize = Point3D(spaceSize.x / 2, spaceSize.y / 2, spaceSize.z / 2);
	this->stepSize = Point3D(spaceSize.x / nCells, spaceSize.y / nCells, spaceSize.z / nCells);
   minPotential = 1 / (float) (1 + (radius * radius));
	potentialBuf = new float[nCells * nCells * nCells];
	memset(potentialBuf, 0, nCells * nCells * nCells * sizeof(float));
	return;
}

Point3D SphereSurface::getIntersection(Point3D p1, Point3D p2)
{
   Point3D t;
   float v1 = getPotential(p1), v2 = getPotential(p2), mp = minPotential;
   t.x = (int) (p1.x + (mp - v1) * (p2.x - p1.x) / (float) (v2 - v1));
   t.y = (int) (p1.y + (mp - v1) * (p2.y - p1.y) / (float) (v2 - v1)); 
   t.z = (int) (p1.z + (mp - v1) * (p2.z - p1.z) / (float) (v2 - v1)); 
   return t;
}

bool SphereSurface::outside(Point3D p)
{
  	if(getPotential(p) < minPotential) return 1;
  	return 0;
}

float SphereSurface::getPotential(Point3D p)
{
   if(p.x < -spaceSize.x || p.y < -spaceSize.y || p.z < -spaceSize.z || p.x >= spaceSize.x || p.y >= spaceSize.y || p.z >= spaceSize.z) return -1;
   int ind = ((p.z + spaceSize.z) / stepSize.z) * nCells * nCells + ((p.y + spaceSize.y) / stepSize.y) * nCells + ((p.x + spaceSize.x) / stepSize.x);
   if(potentialBuf[ind]) return potentialBuf[ind];
  	float total = 0;
   for(int i = 0; i < nSpheres; i++)
      total += 1 / (float) (1 + (
 		((p.x - coords[i].x) * (p.x - coords[i].x)) +
  		((p.y - coords[i].y) * (p.y - coords[i].y)) +
  		((p.z - coords[i].z) * (p.z - coords[i].z)))); 
   potentialBuf[ind] = total;
   return total;
}

SphereSurface::~SphereSurface()
{
	delete [] potentialBuf;
	return;
}

void setGLOptions()
{	
	GLfloat fogColour[] = { 0.0f, 0.2f, 0.2f, 0.0f };
	glEnable(GL_FOG);
	glFogi(GL_FOG_MODE, GL_EXP2);
	glFogfv(GL_FOG_COLOR, fogColour);
	glFogf(GL_FOG_DENSITY, 0.009f);
	glClearColor(fogColour[0], fogColour[1], fogColour[2], 0.0f);		
	glEnable(GL_CULL_FACE);
   glShadeModel(GL_FLAT);
   return;
}